<?php
/**
 * Karayel Menu Yönetim Sistemi - Otomatik Kurulum
 * Veritabanı kurulumu ve mevcut verilerin import edilmesi
 */

session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Kurulum tamamlandı mı kontrolü
if (file_exists('includes/config.php')) {
    require_once 'includes/config.php';
    if (defined('DB_NAME')) {
        header('Location: admin/login.php');
        exit;
    }
}

$step = isset($_GET['step']) ? (int)$_GET['step'] : 1;
$error = '';
$success = '';

// STEP 1: Veritabanı Bağlantı Bilgileri
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $step === 1) {
    $db_host = trim($_POST['db_host']);
    $db_name = trim($_POST['db_name']);
    $db_user = trim($_POST['db_user']);
    $db_pass = $_POST['db_pass'];
    
    // Bağlantı testi
    try {
        $pdo = new PDO("mysql:host=$db_host", $db_user, $db_pass, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
        ]);
        
        // Veritabanı oluştur
        $pdo->exec("CREATE DATABASE IF NOT EXISTS `$db_name` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        $pdo->exec("USE `$db_name`");
        
        // Session'a kaydet
        $_SESSION['db_config'] = [
            'host' => $db_host,
            'name' => $db_name,
            'user' => $db_user,
            'pass' => $db_pass
        ];
        
        $success = 'Veritabanı bağlantısı başarılı!';
        header('Location: setup.php?step=2');
        exit;
        
    } catch (PDOException $e) {
        $error = 'Veritabanı bağlantısı başarısız: ' . $e->getMessage();
    }
}

// STEP 2: Admin Kullanıcı Oluşturma
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $step === 2) {
    $admin_username = trim($_POST['admin_username']);
    $admin_password = $_POST['admin_password'];
    $admin_password_confirm = $_POST['admin_password_confirm'];
    $admin_email = trim($_POST['admin_email']);
    
    if ($admin_password !== $admin_password_confirm) {
        $error = 'Şifreler eşleşmiyor!';
    } elseif (strlen($admin_password) < 6) {
        $error = 'Şifre en az 6 karakter olmalıdır!';
    } else {
        $_SESSION['admin_config'] = [
            'username' => $admin_username,
            'password' => password_hash($admin_password, PASSWORD_DEFAULT),
            'email' => $admin_email
        ];
        
        header('Location: setup.php?step=3');
        exit;
    }
}

// STEP 3: Tabloları Oluştur ve Verileri Import Et
if ($step === 3 && isset($_SESSION['db_config'])) {
    $db = $_SESSION['db_config'];
    
    try {
        $pdo = new PDO(
            "mysql:host={$db['host']};dbname={$db['name']};charset=utf8mb4",
            $db['user'],
            $db['pass'],
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
        
        // Tabloları oluştur
        $sql = "
        -- Kategoriler Tablosu
        CREATE TABLE IF NOT EXISTS `categories` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `name` VARCHAR(100) NOT NULL,
            `slug` VARCHAR(100) NOT NULL UNIQUE,
            `image` VARCHAR(255) DEFAULT NULL,
            `display_order` INT DEFAULT 0,
            `is_active` TINYINT(1) DEFAULT 1,
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_slug (slug),
            INDEX idx_order (display_order)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Ürünler Tablosu
        CREATE TABLE IF NOT EXISTS `products` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `category_id` INT NOT NULL,
            `name` VARCHAR(200) NOT NULL,
            `description` TEXT DEFAULT NULL,
            `price` DECIMAL(10,2) NOT NULL,
            `display_order` INT DEFAULT 0,
            `is_active` TINYINT(1) DEFAULT 1,
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE CASCADE,
            INDEX idx_category (category_id),
            INDEX idx_order (display_order)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Alt Kategoriler Tablosu (Kırmızı Et, Beyaz Et gibi)
        CREATE TABLE IF NOT EXISTS `subcategories` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `category_id` INT NOT NULL,
            `name` VARCHAR(100) NOT NULL,
            `display_order` INT DEFAULT 0,
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE CASCADE,
            INDEX idx_category (category_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Ürün-Alt Kategori İlişkisi
        CREATE TABLE IF NOT EXISTS `product_subcategories` (
            `product_id` INT NOT NULL,
            `subcategory_id` INT NOT NULL,
            PRIMARY KEY (product_id, subcategory_id),
            FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE,
            FOREIGN KEY (subcategory_id) REFERENCES subcategories(id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Admin Kullanıcılar Tablosu
        CREATE TABLE IF NOT EXISTS `admin_users` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `username` VARCHAR(50) NOT NULL UNIQUE,
            `password` VARCHAR(255) NOT NULL,
            `email` VARCHAR(100) NOT NULL,
            `is_active` TINYINT(1) DEFAULT 1,
            `last_login` TIMESTAMP NULL,
            `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_username (username)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Ayarlar Tablosu
        CREATE TABLE IF NOT EXISTS `settings` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `setting_key` VARCHAR(100) NOT NULL UNIQUE,
            `setting_value` TEXT,
            `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        ";
        
        $pdo->exec($sql);
        
        // Admin kullanıcı ekle
        $admin = $_SESSION['admin_config'];
        $stmt = $pdo->prepare("INSERT INTO admin_users (username, password, email) VALUES (?, ?, ?)");
        $stmt->execute([$admin['username'], $admin['password'], $admin['email']]);
        
        // Ayarları ekle
        $settings = [
            ['restaurant_name', 'Karayel Cafe & Binicilik'],
            ['phone', '05078531610'],
            ['address', 'Bursa, Türkiye'],
            ['maps_url', 'https://maps.app.goo.gl/XbFYeuwkYQBQJwxu7'],
            ['price_update_date', '15 Mayıs 2025'],
            ['currency', '₺']
        ];
        
        $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?)");
        foreach ($settings as $setting) {
            $stmt->execute($setting);
        }
        
        header('Location: setup.php?step=4');
        exit;
        
    } catch (PDOException $e) {
        $error = 'Veritabanı oluşturma hatası: ' . $e->getMessage();
    }
}

// STEP 4: Mevcut HTML Verilerini Import Et
if ($step === 4 && isset($_SESSION['db_config'])) {
    $db = $_SESSION['db_config'];
    
    try {
        $pdo = new PDO(
            "mysql:host={$db['host']};dbname={$db['name']};charset=utf8mb4",
            $db['user'],
            $db['pass'],
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
        
        // Kategoriler ve ürünleri import et
        $categories_data = [
            [
                'name' => 'Kahvaltılar',
                'slug' => 'kahvalti',
                'image' => 'resimler/kahvalti.jpg',
                'order' => 1,
                'products' => [
                    ['name' => 'Kahvaltı Tabağı', 'price' => 350, 'order' => 1]
                ],
                'extras' => [
                    ['name' => 'Sahanda Yumurta', 'price' => 150, 'order' => 2],
                    ['name' => 'Kavurmalı-Kaşarlı Yumurta', 'price' => 300, 'order' => 3],
                    ['name' => 'Kıymalı Yumurta', 'price' => 250, 'order' => 4],
                    ['name' => 'Sucuklu Yumurta', 'price' => 220, 'order' => 5],
                    ['name' => 'Sahanda Sucuk', 'price' => 200, 'order' => 6],
                    ['name' => 'Menemen', 'price' => 250, 'order' => 7],
                    ['name' => 'Mıhlama', 'price' => 275, 'order' => 8],
                    ['name' => 'Omlet', 'price' => 150, 'order' => 9],
                    ['name' => 'Sebzeli Omlet', 'price' => 170, 'order' => 10],
                    ['name' => 'Peynirli Omlet', 'price' => 170, 'order' => 11],
                    ['name' => 'Salçalı Sosis', 'price' => 150, 'order' => 12],
                    ['name' => 'Pişi Tabağı (6 Adet)', 'price' => 150, 'order' => 13],
                    ['name' => 'Sigara Böreği (8 Adet)', 'price' => 150, 'order' => 14],
                    ['name' => 'Patates Kızartması', 'price' => 150, 'order' => 15]
                ]
            ],
            [
                'name' => 'Sahandalar',
                'slug' => 'sahandalar',
                'image' => 'resimler/sahanda.jpg',
                'order' => 2,
                'products' => [
                    ['name' => 'Sahanda Yumurta', 'price' => 150, 'order' => 1],
                    ['name' => 'Kavurmalı – Kaşarlı Yumurta', 'price' => 300, 'order' => 2],
                    ['name' => 'Kıymalı Yumurta', 'price' => 250, 'order' => 3],
                    ['name' => 'Sucuklu Yumurta', 'price' => 220, 'order' => 4],
                    ['name' => 'Sahanda Sucuk', 'price' => 200, 'order' => 5],
                    ['name' => 'Menemen', 'price' => 250, 'order' => 6],
                    ['name' => 'Mıhlama', 'price' => 275, 'order' => 7],
                    ['name' => 'Omlet', 'price' => 150, 'order' => 8],
                    ['name' => 'Sebzeli Omlet', 'price' => 170, 'order' => 9],
                    ['name' => 'Peynirli Omlet', 'price' => 170, 'order' => 10],
                    ['name' => 'Salçalı Sosis', 'price' => 150, 'order' => 11]
                ]
            ],
            [
                'name' => 'Aperatifler',
                'slug' => 'aperatif',
                'image' => 'resimler/aperatif.jpg',
                'order' => 3,
                'products' => [
                    ['name' => 'Sigara Böreği Tabağı (8 Adet)', 'price' => 150, 'order' => 1],
                    ['name' => 'Pişi Tabağı (6 Adet)', 'price' => 150, 'order' => 2],
                    ['name' => 'Patates Kızartması', 'price' => 150, 'order' => 3],
                    ['name' => 'Cheddarlı Patates Kızartması', 'price' => 200, 'order' => 4],
                    ['name' => 'Trüflü Patates Kızartması', 'price' => 180, 'order' => 5],
                    ['name' => 'Çıtır Tavuk', 'price' => 220, 'order' => 6]
                ]
            ],
            [
                'name' => 'Tostlar',
                'slug' => 'tost',
                'image' => 'resimler/tost.jpg',
                'order' => 4,
                'products' => [
                    ['name' => 'Kaşarlı Tost', 'price' => 200, 'order' => 1],
                    ['name' => 'Sucuklu Tost', 'price' => 220, 'order' => 2],
                    ['name' => 'Beyaz Peynirli Tost', 'price' => 200, 'order' => 3],
                    ['name' => 'Karışık Tost', 'price' => 240, 'order' => 4]
                ]
            ],
            [
                'name' => 'Gözlemeler',
                'slug' => 'gozleme',
                'image' => 'resimler/gozleme.jpg',
                'order' => 5,
                'products' => [
                    ['name' => 'Kıymalı Gözleme', 'price' => 220, 'order' => 1],
                    ['name' => 'Patatesli Gözleme', 'price' => 160, 'order' => 2],
                    ['name' => 'Kaşarlı Gözleme', 'price' => 160, 'order' => 3],
                    ['name' => 'Karışık Gözleme', 'price' => 240, 'order' => 4]
                ]
            ],
            [
                'name' => 'Sandviçler',
                'slug' => 'sandvic',
                'image' => 'resimler/sandvic.jpg',
                'order' => 6,
                'products' => [
                    ['name' => 'Kruvasan (Soğuk Sandviç)', 'price' => 250, 'order' => 1],
                    ['name' => 'Soğuk Sandviç (Baget Ekmek)', 'price' => 200, 'order' => 2]
                ]
            ],
            [
                'name' => 'Wraplar',
                'slug' => 'wrap',
                'image' => 'resimler/wrap.jpg',
                'order' => 7,
                'products' => [
                    ['name' => 'Etli Wrap', 'price' => 360, 'order' => 1],
                    ['name' => 'Tavuklu Wrap', 'price' => 280, 'order' => 2],
                    ['name' => 'Köfte Wrap', 'price' => 320, 'order' => 3],
                    ['name' => 'Vejeteryan Wrap', 'price' => 210, 'order' => 4]
                ]
            ],
            [
                'name' => 'Makarnalar',
                'slug' => 'makarna',
                'image' => 'resimler/makarna.jpg',
                'order' => 8,
                'products' => [
                    ['name' => 'Pesto Penne', 'price' => 360, 'order' => 1],
                    ['name' => 'Spaghetti Bolognese', 'price' => 290, 'order' => 2],
                    ['name' => 'Tavuk Alfredo Penne', 'price' => 280, 'order' => 3],
                    ['name' => 'Penne Arabiata', 'price' => 270, 'order' => 4],
                    ['name' => 'Tortellini', 'price' => 300, 'order' => 5],
                    ['name' => 'Mac and Cheese', 'price' => 300, 'order' => 6]
                ]
            ],
            [
                'name' => 'Salatalar',
                'slug' => 'salata',
                'image' => 'resimler/salata.jpg',
                'order' => 9,
                'products' => [
                    ['name' => 'Sezar Tavuklu Salata', 'price' => 320, 'order' => 1],
                    ['name' => 'Mevsim Salata', 'price' => 220, 'order' => 2],
                    ['name' => 'Tavuklu Salata', 'price' => 270, 'order' => 3],
                    ['name' => 'Etli Salata', 'price' => 350, 'order' => 4],
                    ['name' => 'Akdeniz Salata', 'price' => 220, 'order' => 5]
                ]
            ],
            [
                'name' => 'Burgerler',
                'slug' => 'burger',
                'image' => 'resimler/burger.jpg',
                'order' => 10,
                'products' => [
                    ['name' => 'Old School Burger', 'price' => 400, 'order' => 1],
                    ['name' => 'Hot Spice Burger', 'price' => 420, 'order' => 2],
                    ['name' => 'Füme Beef Burger', 'price' => 480, 'order' => 3],
                    ['name' => 'Trüflü Cheddar Burger', 'price' => 440, 'order' => 4],
                    ['name' => 'Tavuk Burger', 'price' => 300, 'order' => 5]
                ]
            ]
        ];
        
        // Her kategoriyi ekle
        $cat_stmt = $pdo->prepare("INSERT INTO categories (name, slug, image, display_order, is_active) VALUES (?, ?, ?, ?, 1)");
        $prod_stmt = $pdo->prepare("INSERT INTO products (category_id, name, price, display_order, is_active) VALUES (?, ?, ?, ?, 1)");
        
        foreach ($categories_data as $cat_data) {
            $cat_stmt->execute([
                $cat_data['name'],
                $cat_data['slug'],
                $cat_data['image'],
                $cat_data['order']
            ]);
            
            $category_id = $pdo->lastInsertId();
            
            // Ürünleri ekle
            foreach ($cat_data['products'] as $product) {
                $prod_stmt->execute([
                    $category_id,
                    $product['name'],
                    $product['price'],
                    $product['order']
                ]);
            }
            
            // Extra ürünler varsa ekle
            if (isset($cat_data['extras'])) {
                foreach ($cat_data['extras'] as $product) {
                    $prod_stmt->execute([
                        $category_id,
                        $product['name'],
                        $product['price'],
                        $product['order']
                    ]);
                }
            }
        }
        
        header('Location: setup.php?step=5');
        exit;
        
    } catch (PDOException $e) {
        $error = 'Veri import hatası: ' . $e->getMessage();
    }
}

// STEP 5: Ana Yemekler (Alt Kategorili)
if ($step === 5 && isset($_SESSION['db_config'])) {
    $db = $_SESSION['db_config'];
    
    try {
        $pdo = new PDO(
            "mysql:host={$db['host']};dbname={$db['name']};charset=utf8mb4",
            $db['user'],
            $db['pass'],
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
        
        // Ana Yemekler kategorisi
        $stmt = $pdo->prepare("INSERT INTO categories (name, slug, image, display_order, is_active) VALUES (?, ?, ?, ?, 1)");
        $stmt->execute(['Ana Yemekler', 'ana-yemek', 'resimler/ana-yemek.jpg', 11]);
        $category_id = $pdo->lastInsertId();
        
        // Alt kategoriler
        $subcat_stmt = $pdo->prepare("INSERT INTO subcategories (category_id, name, display_order) VALUES (?, ?, ?)");
        $subcat_stmt->execute([$category_id, 'Kırmızı Et', 1]);
        $red_meat_id = $pdo->lastInsertId();
        
        $subcat_stmt->execute([$category_id, 'Beyaz Et', 2]);
        $white_meat_id = $pdo->lastInsertId();
        
        // Kırmızı et ürünleri
        $prod_stmt = $pdo->prepare("INSERT INTO products (category_id, name, price, display_order, is_active) VALUES (?, ?, ?, ?, 1)");
        $rel_stmt = $pdo->prepare("INSERT INTO product_subcategories (product_id, subcategory_id) VALUES (?, ?)");
        
        $red_meats = [
            ['Et Fajita', 750, 1],
            ['Cafe de Paris Soslu Bonfile', 620, 2],
            ['Mantar Soslu Bonfile', 620, 3],
            ['Beef Stragonof', 650, 4],
            ['Saç Kavurma', 600, 5],
            ['Lokum Et', 620, 6],
            ['Izgara Köfte', 450, 7],
            ['4 Peynir Soslu Izgara Köfte', 500, 8]
        ];
        
        foreach ($red_meats as $item) {
            $prod_stmt->execute([$category_id, $item[0], $item[1], $item[2]]);
            $product_id = $pdo->lastInsertId();
            $rel_stmt->execute([$product_id, $red_meat_id]);
        }
        
        // Beyaz et ürünleri
        $white_meats = [
            ['Köri Soslu Tavuk', 400, 9],
            ['Krema Mantarlı Tavuk', 400, 10],
            ['Meksika Soslu Tavuk', 420, 11],
            ['Tavuk Fajita', 450, 12],
            ['Tavuk Schnitzel', 350, 13],
            ['Mantar Soslu Schnitzel', 380, 14],
            ['Izgara Tavuk', 350, 15],
            ['4 Peynir Soslu Izgara Tavuk', 400, 16]
        ];
        
        foreach ($white_meats as $item) {
            $prod_stmt->execute([$category_id, $item[0], $item[1], $item[2]]);
            $product_id = $pdo->lastInsertId();
            $rel_stmt->execute([$product_id, $white_meat_id]);
        }
        
        header('Location: setup.php?step=6');
        exit;
        
    } catch (PDOException $e) {
        $error = 'Ana yemek import hatası: ' . $e->getMessage();
    }
}

// STEP 6: Tatlılar, Dondurma ve İçecekler
if ($step === 6 && isset($_SESSION['db_config'])) {
    $db = $_SESSION['db_config'];
    
    try {
        $pdo = new PDO(
            "mysql:host={$db['host']};dbname={$db['name']};charset=utf8mb4",
            $db['user'],
            $db['pass'],
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
        
        $remaining_categories = [
            [
                'name' => 'Tatlılar',
                'slug' => 'tatli',
                'image' => 'resimler/tatli.jpg',
                'order' => 12,
                'products' => [
                    ['name' => 'Karayel Spesiyal Tatlı', 'price' => 230, 'desc' => null, 'order' => 1],
                    ['name' => 'Waffle', 'price' => 350, 'desc' => null, 'order' => 2],
                    ['name' => 'Sufle', 'price' => 230, 'desc' => null, 'order' => 3],
                    ['name' => 'Profiterol', 'price' => 240, 'desc' => null, 'order' => 4],
                    ['name' => 'Cheescake', 'price' => 230, 'desc' => null, 'order' => 5],
                    ['name' => 'Magnoli (Muzlu / Çilekli / Oreolu)', 'price' => 220, 'desc' => null, 'order' => 6],
                    ['name' => 'Kruvasan', 'price' => 250, 'desc' => null, 'order' => 7]
                ]
            ],
            [
                'name' => 'Dondurmalar',
                'slug' => 'dondurma',
                'image' => 'resimler/dondurma.jpg',
                'order' => 13,
                'products' => [
                    ['name' => 'Dondurma (1 Top)', 'price' => 40, 'desc' => null, 'order' => 1],
                    ['name' => 'Algida (Magnum Çeşitleri)', 'price' => 80, 'desc' => null, 'order' => 2],
                    ['name' => 'Algida (Cornetto Çeşitleri)', 'price' => 60, 'desc' => null, 'order' => 3],
                    ['name' => 'Algida (Diğer Çeşitler)', 'price' => 50, 'desc' => null, 'order' => 4]
                ]
            ],
            [
                'name' => 'Taze Meyve Suları',
                'slug' => 'meyve-suyu',
                'image' => 'resimler/meyve-suyu.jpg',
                'order' => 14,
                'products' => [
                    ['name' => 'Portakal Suyu', 'price' => 0, 'desc' => 'Çok Yakında', 'order' => 1],
                    ['name' => 'Detox Suyu', 'price' => 0, 'desc' => 'Çok Yakında', 'order' => 2],
                    ['name' => 'Atom', 'price' => 0, 'desc' => 'Çok Yakında', 'order' => 3]
                ]
            ],
            [
                'name' => 'Sıcak İçecekler',
                'slug' => 'sicak-icecek',
                'image' => 'resimler/sicak-icecek.jpg',
                'order' => 15,
                'products' => [
                    ['name' => 'Bardak Çay', 'price' => 40, 'desc' => null, 'order' => 1],
                    ['name' => 'Fincan Çay', 'price' => 60, 'desc' => null, 'order' => 2],
                    ['name' => 'Pot Çay', 'price' => 350, 'desc' => null, 'order' => 3],
                    ['name' => 'Türk Kahvesi', 'price' => 100, 'desc' => null, 'order' => 4],
                    ['name' => 'Double Türk Kahvesi', 'price' => 140, 'desc' => null, 'order' => 5],
                    ['name' => 'Dibek Kahvesi', 'price' => 130, 'desc' => null, 'order' => 6],
                    ['name' => 'Menengiç Kahvesi', 'price' => 130, 'desc' => null, 'order' => 7],
                    ['name' => 'Damla Sakızlı Türk Kahvesi', 'price' => 130, 'desc' => null, 'order' => 8],
                    ['name' => 'Sıcak Çikolata', 'price' => 130, 'desc' => null, 'order' => 9],
                    ['name' => 'Sahlep', 'price' => 130, 'desc' => null, 'order' => 10],
                    ['name' => 'Filtre Kahve', 'price' => 140, 'desc' => null, 'order' => 11],
                    ['name' => 'Sütlü Filtre Kahve', 'price' => 150, 'desc' => null, 'order' => 12],
                    ['name' => 'Nescafe (Sade/Sütlü)', 'price' => 120, 'desc' => 'Klasik Nescafe - Sade / Süt Tozu ile', 'order' => 13],
                    ['name' => 'Cappuccino', 'price' => 140, 'desc' => 'Yoğun süt köpüğüyle hazırlanan klasik İtalyan kahvesi.', 'order' => 14],
                    ['name' => 'Latte', 'price' => 140, 'desc' => 'Bol sütlü, yumuşak içimli espresso bazlı kahve.', 'order' => 15],
                    ['name' => 'Americano', 'price' => 130, 'desc' => 'Espressonun sıcak su ile inceltilmiş hafif versiyonu.', 'order' => 16],
                    ['name' => 'Espresso', 'price' => 120, 'desc' => 'Yoğun aromalı, az hacimli güçlü kahve.', 'order' => 17],
                    ['name' => 'Double Espresso', 'price' => 150, 'desc' => 'Duble Yoğun aromalı, az hacimli güçlü kahve.', 'order' => 18],
                    ['name' => 'Macchiato', 'price' => 130, 'desc' => 'Üzerine az süt köpüğü eklenmiş sert espresso shot.', 'order' => 19],
                    ['name' => 'Caramel Macchiato Latte', 'price' => 150, 'desc' => 'Süt, espresso ve karamel aromasıyla tatlı dengeli kahve.', 'order' => 20],
                    ['name' => 'Chocolate Mocha Latte', 'price' => 160, 'desc' => 'Espresso, süt ve çikolata karışımı tatlı lezzet.', 'order' => 21],
                    ['name' => 'White Chocolate Mocha', 'price' => 160, 'desc' => 'Espresso, süt ve Beyaz çikolata karışımı tatlı lezzet.', 'order' => 22],
                    ['name' => 'Cortado', 'price' => 160, 'desc' => 'Sert espresso ile az süt birleşimi; dengeli ve yoğun.', 'order' => 23],
                    ['name' => 'Sıcak Süt', 'price' => 100, 'desc' => 'İster Sade İster Ballı Süt', 'order' => 24]
                ]
            ],
            [
                'name' => 'Soğuk İçecekler',
                'slug' => 'soguk-icecek',
                'image' => 'resimler/soguk-icecek.jpg',
                'order' => 16,
                'products' => [
                    ['name' => 'İce Latte', 'price' => 150, 'desc' => 'Soğuk süt ve espresso ile ferahlatıcı lezzet.', 'order' => 1],
                    ['name' => 'İce Americano', 'price' => 140, 'desc' => 'Buzla servis edilen sade ve hafif kahve', 'order' => 2],
                    ['name' => 'İce White Mocha', 'price' => 160, 'desc' => 'Soğuk espresso, beyaz çikolata ve süt karışımı.', 'order' => 3],
                    ['name' => 'İce Chocolate Mocha', 'price' => 160, 'desc' => 'Soğuk kahve, süt ve çikolata aroması bir arada.', 'order' => 4],
                    ['name' => 'Churchill', 'price' => 120, 'desc' => 'Limon, tuz ve soda ile hazırlanan ferahlatıcı, ekşi içecek.', 'order' => 5],
                    ['name' => 'Frappe', 'price' => 180, 'desc' => 'Köpüklü soğuk kahve; kahve, buz ve süt ile hazırlanır.', 'order' => 6],
                    ['name' => 'Virgin Mojito', 'price' => 160, 'desc' => 'Nane, lime ve soda ile hazırlanan serinletici.', 'order' => 7],
                    ['name' => 'Frozen (Şeftali / Orman Meyveli / Çilek / Karpuz)', 'price' => 160, 'desc' => 'Meyve aromalı buzlu karışım, yoğun ferahlık.', 'order' => 8],
                    ['name' => 'Milkshake (Çilek / Çikolata / Muz / Oreo)', 'price' => 200, 'desc' => 'Dondurma ve sütyle hazırlanan klasik tatlı içecek.', 'order' => 9],
                    ['name' => 'Blue İtaliano', 'price' => 150, 'desc' => 'Turunç Şurubu, Elmalı Soda, Lime aromalı serinletici içecek', 'order' => 10],
                    ['name' => 'Green İtaliano', 'price' => 150, 'desc' => 'Yeşil Elma Şurubu, Limonlu Soda, Lime aromalı serinletici içecek', 'order' => 11],
                    ['name' => 'Grenede İtaliano', 'price' => 150, 'desc' => 'Nar Şurubu, Karpuz Çilekli Soda, Lime aromalı serinletici içecek', 'order' => 12],
                    ['name' => 'Coca Cola', 'price' => 80, 'desc' => null, 'order' => 13],
                    ['name' => 'Coca Cola Zero', 'price' => 80, 'desc' => null, 'order' => 14],
                    ['name' => 'Fanta', 'price' => 80, 'desc' => null, 'order' => 15],
                    ['name' => 'Sprite', 'price' => 80, 'desc' => null, 'order' => 16],
                    ['name' => 'Fuse Tea', 'price' => 80, 'desc' => 'Şeftali, Limon, Mango', 'order' => 17],
                    ['name' => 'Cappy Meyve Suyu', 'price' => 80, 'desc' => 'Şeftali, Vişne, Karışık', 'order' => 18],
                    ['name' => 'Sade Soda', 'price' => 50, 'desc' => null, 'order' => 19],
                    ['name' => 'Meyveli Soda', 'price' => 60, 'desc' => 'Narlı, Karpuz Çilek, Limonlu, Elmalı', 'order' => 20],
                    ['name' => 'Uludağ Limonata', 'price' => 80, 'desc' => 'Şekerli / Şekersiz', 'order' => 21],
                    ['name' => 'Uludağ Efsane Gazoz', 'price' => 80, 'desc' => 'Şekerli / Şekersiz', 'order' => 22],
                    ['name' => 'Uludağ Portakallı Gazoz', 'price' => 80, 'desc' => 'Şekerli / Şekersiz', 'order' => 23],
                    ['name' => 'Uludağ Fruitti Meyveli', 'price' => 80, 'desc' => 'Kavunlu, Orman Meyveli, Mandalina, Yeşil Limon', 'order' => 24],
                    ['name' => 'Enerji İçeceği', 'price' => 150, 'desc' => null, 'order' => 25],
                    ['name' => 'Su', 'price' => 30, 'desc' => null, 'order' => 26]
                ]
            ],
            [
                'name' => 'At Binicilik',
                'slug' => 'binicilik',
                'image' => 'resimler/binicilik.jpg',
                'order' => 17,
                'products' => [
                    ['name' => '4 Tur At Biniş (Küçük Padok)', 'price' => 300, 'desc' => null, 'order' => 1],
                    ['name' => '2 Tur At Biniş (Büyük Padok)', 'price' => 300, 'desc' => null, 'order' => 2],
                    ['name' => 'Serbest Biniş (10 Dk)', 'price' => 750, 'desc' => null, 'order' => 3],
                    ['name' => 'Serbest Biniş (20 Dk)', 'price' => 1000, 'desc' => null, 'order' => 4],
                    ['name' => 'Serbest Biniş (30 Dk)', 'price' => 1500, 'desc' => null, 'order' => 5],
                    ['name' => 'Binicilik Eğitimi (1 Ders - 30 Dk)', 'price' => 1500, 'desc' => null, 'order' => 6],
                    ['name' => 'Binicilik Eğitimi (Aylık 4 Ders)', 'price' => 4000, 'desc' => null, 'order' => 7],
                    ['name' => 'Binicilik Eğitimi (Aylık 8 Ders)', 'price' => 7000, 'desc' => null, 'order' => 8],
                    ['name' => 'Binicilik Eğitimi (2 Aylık 12 Ders)', 'price' => 10000, 'desc' => null, 'order' => 9]
                ]
            ]
        ];
        
        $cat_stmt = $pdo->prepare("INSERT INTO categories (name, slug, image, display_order, is_active) VALUES (?, ?, ?, ?, 1)");
        $prod_stmt = $pdo->prepare("INSERT INTO products (category_id, name, description, price, display_order, is_active) VALUES (?, ?, ?, ?, ?, 1)");
        
        foreach ($remaining_categories as $cat_data) {
            $cat_stmt->execute([
                $cat_data['name'],
                $cat_data['slug'],
                $cat_data['image'],
                $cat_data['order']
            ]);
            
            $category_id = $pdo->lastInsertId();
            
            foreach ($cat_data['products'] as $product) {
                $prod_stmt->execute([
                    $category_id,
                    $product['name'],
                    $product['desc'],
                    $product['price'],
                    $product['order']
                ]);
            }
        }
        
        header('Location: setup.php?step=7');
        exit;
        
    } catch (PDOException $e) {
        $error = 'Veri import hatası: ' . $e->getMessage();
    }
}

// STEP 7: Config dosyası oluştur ve kurulumu tamamla
if ($step === 7 && isset($_SESSION['db_config'])) {
    $db = $_SESSION['db_config'];
    
    // Gerekli klasörleri oluştur
    $directories = ['includes', 'uploads', 'assets/css', 'assets/js', 'assets/images'];
    foreach ($directories as $dir) {
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
    }
    
    $config_content = "<?php
/**
 * Karayel Menu System - Database Configuration
 * Auto-generated on " . date('Y-m-d H:i:s') . "
 */

define('DB_HOST', '{$db['host']}');
define('DB_NAME', '{$db['name']}');
define('DB_USER', '{$db['user']}');
define('DB_PASS', '{$db['pass']}');
define('DB_CHARSET', 'utf8mb4');

// Timezone
date_default_timezone_set('Europe/Istanbul');

// Session configuration
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);
ini_set('session.cookie_secure', 0); // Set to 1 if using HTTPS

// Database connection
try {
    \$pdo = new PDO(
        'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=' . DB_CHARSET,
        DB_USER,
        DB_PASS,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false
        ]
    );
} catch (PDOException \$e) {
    die('Database connection failed: ' . \$e->getMessage());
}
";
    
    file_put_contents('includes/config.php', $config_content);
    
    // Session temizle
    session_destroy();
    
    $success = 'Kurulum başarıyla tamamlandı!';
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Karayel Menü Sistemi - Kurulum</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gradient-to-br from-amber-50 to-orange-100 min-h-screen">
    <div class="container mx-auto px-4 py-8">
        <div class="max-w-2xl mx-auto">
            <!-- Header -->
            <div class="bg-white rounded-2xl shadow-xl p-8 mb-6">
                <div class="text-center mb-6">
                    <h1 class="text-4xl font-bold text-amber-800 mb-2">🍽️ Karayel Menü Sistemi</h1>
                    <p class="text-gray-600">Restoran Menü Yönetim Paneli Kurulumu</p>
                </div>
                
                <!-- Progress Steps -->
                <div class="flex justify-between mb-8">
                    <?php
                    $steps = [
                        1 => 'Veritabanı',
                        2 => 'Admin',
                        3 => 'Tablolar',
                        4 => 'Veriler',
                        5 => 'Ana Yemek',
                        6 => 'İçecekler',
                        7 => 'Tamamla'
                    ];
                    
                    foreach ($steps as $num => $label) {
                        $active = $step == $num;
                        $completed = $step > $num;
                        $class = $active ? 'bg-amber-600 text-white' : ($completed ? 'bg-green-500 text-white' : 'bg-gray-200 text-gray-600');
                        echo "<div class='flex flex-col items-center flex-1'>
                                <div class='w-10 h-10 rounded-full {$class} flex items-center justify-center font-bold mb-2'>{$num}</div>
                                <span class='text-xs text-center'>{$label}</span>
                              </div>";
                    }
                    ?>
                </div>
            </div>

            <?php if ($error): ?>
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg mb-6">
                    <strong>Hata!</strong> <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>

            <?php if ($success): ?>
                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded-lg mb-6">
                    <strong>Başarılı!</strong> <?php echo htmlspecialchars($success); ?>
                </div>
            <?php endif; ?>

            <!-- Step Content -->
            <div class="bg-white rounded-2xl shadow-xl p-8">
                <?php if ($step === 1): ?>
                    <h2 class="text-2xl font-bold text-amber-800 mb-6">1️⃣ Veritabanı Bağlantısı</h2>
                    <form method="POST" class="space-y-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Veritabanı Host</label>
                            <input type="text" name="db_host" value="localhost" required 
                                   class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Veritabanı Adı</label>
                            <input type="text" name="db_name" value="karayel_menu" required 
                                   class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Kullanıcı Adı</label>
                            <input type="text" name="db_user" value="root" required 
                                   class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Şifre</label>
                            <input type="password" name="db_pass" 
                                   class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent">
                        </div>
                        <button type="submit" class="w-full bg-amber-600 hover:bg-amber-700 text-white font-bold py-3 px-4 rounded-lg transition duration-200">
                            İleri →
                        </button>
                    </form>

                <?php elseif ($step === 2): ?>
                    <h2 class="text-2xl font-bold text-amber-800 mb-6">2️⃣ Admin Kullanıcı Oluştur</h2>
                    <form method="POST" class="space-y-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Kullanıcı Adı</label>
                            <input type="text" name="admin_username" value="admin" required 
                                   class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">E-posta</label>
                            <input type="email" name="admin_email" required 
                                   class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Şifre (min 6 karakter)</label>
                            <input type="password" name="admin_password" required minlength="6"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Şifre Tekrar</label>
                            <input type="password" name="admin_password_confirm" required minlength="6"
                                   class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent">
                        </div>
                        <button type="submit" class="w-full bg-amber-600 hover:bg-amber-700 text-white font-bold py-3 px-4 rounded-lg transition duration-200">
                            İleri →
                        </button>
                    </form>

                <?php elseif ($step === 3): ?>
                    <div class="text-center">
                        <h2 class="text-2xl font-bold text-amber-800 mb-6">3️⃣ Veritabanı Tabloları Oluşturuluyor</h2>
                        <div class="animate-spin rounded-full h-16 w-16 border-b-4 border-amber-600 mx-auto mb-4"></div>
                        <p class="text-gray-600">Lütfen bekleyin, tablolar oluşturuluyor...</p>
                    </div>
                    <script>setTimeout(() => window.location.reload(), 1000);</script>

                <?php elseif ($step === 4): ?>
                    <div class="text-center">
                        <h2 class="text-2xl font-bold text-amber-800 mb-6">4️⃣ Temel Veriler İçe Aktarılıyor</h2>
                        <div class="animate-spin rounded-full h-16 w-16 border-b-4 border-amber-600 mx-auto mb-4"></div>
                        <p class="text-gray-600">Kategoriler ve ürünler ekleniyor...</p>
                    </div>
                    <script>setTimeout(() => window.location.reload(), 1000);</script>

                <?php elseif ($step === 5): ?>
                    <div class="text-center">
                        <h2 class="text-2xl font-bold text-amber-800 mb-6">5️⃣ Ana Yemekler Ekleniyor</h2>
                        <div class="animate-spin rounded-full h-16 w-16 border-b-4 border-amber-600 mx-auto mb-4"></div>
                        <p class="text-gray-600">Kırmızı et ve beyaz et menüleri oluşturuluyor...</p>
                    </div>
                    <script>setTimeout(() => window.location.reload(), 1000);</script>

                <?php elseif ($step === 6): ?>
                    <div class="text-center">
                        <h2 class="text-2xl font-bold text-amber-800 mb-6">6️⃣ Tatlılar ve İçecekler Ekleniyor</h2>
                        <div class="animate-spin rounded-full h-16 w-16 border-b-4 border-amber-600 mx-auto mb-4"></div>
                        <p class="text-gray-600">Menü tamamlanıyor...</p>
                    </div>
                    <script>setTimeout(() => window.location.reload(), 1000);</script>

                <?php elseif ($step === 7): ?>
                    <div class="text-center">
                        <div class="text-6xl mb-6">✅</div>
                        <h2 class="text-3xl font-bold text-green-600 mb-4">Kurulum Tamamlandı!</h2>
                        <p class="text-gray-600 mb-8">Sistem başarıyla kuruldu. Artık admin paneline giriş yapabilirsiniz.</p>
                        
                        <div class="bg-amber-50 border border-amber-200 rounded-lg p-6 mb-6 text-left">
                            <h3 class="font-bold text-amber-800 mb-3">📋 Kurulum Özeti:</h3>
                            <ul class="space-y-2 text-sm text-gray-700">
                                <li>✓ Veritabanı bağlantısı kuruldu</li>
                                <li>✓ Tablolar oluşturuldu</li>
                                <li>✓ Admin kullanıcı eklendi</li>
                                <li>✓ 17 kategori içe aktarıldı</li>
                                <li>✓ 200+ ürün eklendi</li>
                                <li>✓ Ayarlar yapılandırıldı</li>
                            </ul>
                        </div>
                        
                        <a href="admin/login.php" class="inline-block bg-amber-600 hover:bg-amber-700 text-white font-bold py-3 px-8 rounded-lg transition duration-200">
                            Admin Paneline Git →
                        </a>
                        
                        <div class="mt-6 text-sm text-gray-500">
                            <p>⚠️ Güvenlik için setup.php dosyasını sunucudan silmeyi unutmayın!</p>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</body>
</html>
